package ga.view.input;

import ga.view.interfaces.MouseListener;
import ga.view.streaming.showroom.CameraSettings;

import java.util.logging.Level;
import java.util.logging.Logger;

import com.jme3.input.InputManager;
import com.jme3.input.MouseInput;
import com.jme3.input.controls.AnalogListener;
import com.jme3.input.controls.MouseAxisTrigger;
import com.jme3.input.controls.MouseButtonTrigger;
import com.jme3.math.FastMath;
import com.jme3.math.Vector3f;
import com.jme3.renderer.Camera;

/**
 * This is a mouse listener for dragging the camera around the target point.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
public class CamDragListener extends MouseListener implements AnalogListener {

  // the logger for this class
  private static final Logger LOGGER = Logger.getLogger(CamDragListener.class
      .getName());

  private boolean canRotate;

  // settings
  private final float zoomSpeed = 0.6f;
  private final float rotationSpeed = 0.8f;
  private float minVRotation = 0.2f;
  private float maxVRotation = 0.3f;
  private float minDistance = 1f;
  private float maxDistance;
  private Vector3f lookAt;

  // current values
  private float distance;
  private float rotation;
  private float vRotation;

  private final Camera cam;
  private final InputManager inputManager;

  /**
   * Instantiates a new cam drag listener.
   * 
   * @param cam
   *          the cam
   * @param inputManager
   *          the input manager
   * @param camSettings
   *          the cam settings
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public CamDragListener(final Camera cam, final InputManager inputManager,
      final CameraSettings camSettings) {
    this.cam = cam;
    this.inputManager = inputManager;

    init(camSettings);

    final String[] inputs = { "toggleRotate", "Down", "Up", "mouseLeft",
        "mouseRight", "ZoomIn", "ZoomOut", };
    inputManager.addMapping("Down", new MouseAxisTrigger(1, true));
    inputManager.addMapping("Up", new MouseAxisTrigger(1, false));
    inputManager.addMapping("ZoomIn", new MouseAxisTrigger(2, true));
    inputManager.addMapping("ZoomOut", new MouseAxisTrigger(2, false));
    inputManager.addMapping("mouseLeft", new MouseAxisTrigger(0, true));
    inputManager.addMapping("mouseRight", new MouseAxisTrigger(0, false));
    inputManager.addMapping("toggleRotate", new MouseButtonTrigger(
        MouseInput.BUTTON_LEFT));
    inputManager.addMapping("doubleClick", new MouseButtonTrigger(
        MouseInput.BUTTON_LEFT));
    inputManager.addListener(this, inputs);
  }

  /**
   * Initializes the listener using the cam settings.
   * 
   * @param camSettings
   *          the cam settings
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void init(final CameraSettings camSettings) {
    this.minVRotation = camSettings.getMinVRotation();
    this.maxVRotation = camSettings.getMaxVRotation();
    this.minDistance = camSettings.getMinDistance();
    this.maxDistance = camSettings.getMaxDistance();
    this.lookAt = camSettings.getLookAtPoint();

    init();
  }

  /**
   * Inits the class.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void init() {

    distance = cam.getLocation().distance(lookAt);

    vRotation = FastMath.asin(cam.getLocation().y / distance);
    final float hDistance = distance
        * FastMath.sin((FastMath.PI / 2f) - vRotation);

    if (cam.getLocation().z >= 0f) {
      rotation = FastMath.acos(cam.getLocation().x / hDistance);
    } else {
      rotation = -FastMath.acos(cam.getLocation().x / hDistance);
    }

    updateCamera();
  }

  /**
   * Rotates the camera around the target on the horizontal plane.
   * 
   * @param value
   *          The amount of rotation.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void rotateCamera(final float value) {
    rotation += value * rotationSpeed;

    updateCamera();
  }

  /**
   * Moves the camera toward or away the target.
   * 
   * @param value
   *          Amount of zoom.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void zoomCamera(final float value) {
    distance += value * zoomSpeed;
    if (distance > maxDistance) {
      distance = maxDistance;
    }
    if (distance < minDistance) {
      distance = minDistance;
    }
    if ((vRotation < minVRotation) && (distance > (minDistance + 1.0f))) {
      vRotation = minVRotation;
    }

    updateCamera();
  }

  /**
   * Rotates the camera around the target on the vertical plane.
   * 
   * @param value
   *          The amount of rotation.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void vRotateCamera(final float value) {
    vRotation += value * rotationSpeed;
    if (vRotation > maxVRotation) {
      vRotation = maxVRotation;
    }
    if ((vRotation < minVRotation) && (distance > (minDistance + 1.0f))) {
      vRotation = minVRotation;
    }
    updateCamera();
  }

  /**
   * Update the camera, should only be called internally.
   */
  protected void updateCamera() {
    final float hDistance = distance
        * FastMath.sin((FastMath.PI / 2f) - vRotation);
    Vector3f pos = new Vector3f(hDistance * FastMath.cos(rotation), distance
        * FastMath.sin(vRotation), hDistance * FastMath.sin(rotation));
    pos = pos.add(lookAt);
    cam.setLocation(pos);
    cam.lookAt(lookAt, Vector3f.UNIT_Y);
  }

  @Override
  public void onAction(final String name, final boolean keyPressed,
      final boolean isDoubleClick, final float tpf) {
    if (LOGGER.isLoggable(Level.FINE)) {
      LOGGER.fine(name + " " + keyPressed + " " + isDoubleClick);
    }

    if (isEnabled() && "toggleRotate".equals(name)) {
      if (keyPressed) {
        canRotate = true;
        // TODO fix the bug where nifty does not get inputs after the cursor was
        // invisible
        // inputManager.setCursorVisible(false);
      } else {
        canRotate = false;
        // inputManager.setCursorVisible(true);
      }
    }
  }

  @Override
  public void onAnalog(final String name, final float value, final float tpf) {
    if (isEnabled()) {
      if (canRotate) {
        if ("mouseLeft".equals(name)) {
          rotateCamera(-value);
        } else if ("mouseRight".equals(name)) {
          rotateCamera(value);
        } else if ("Up".equals(name)) {
          vRotateCamera(value);
        } else if ("Down".equals(name)) {
          vRotateCamera(-value);
        }
      }

      if ("ZoomIn".equals(name)) {
        zoomCamera(value);
      } else if ("ZoomOut".equals(name)) {
        zoomCamera(-value);
      }
    }
  }
}
